//
//  HNSBasicCompiler.m
//  CocoaBasic
//
//  Created by Dr. H. Nikolaus Schaller on Tue Oct 08 2002.
//  Copyright (c) 2002 __MyCompanyName__. All rights reserved.
//

#import <HNSAppKit/HNSBasicCompiler.h>

@interface HNSBasicCompiler (PrivateExtension)

- (void) scanStatement;							// scan (and compile) full statement and append to code
- (NSString *) scanName;						// scan symbolic name
- (void) scanParameterList;						// scan method and parameter list
#define Object_Name		@"Name"					// name of variable
#define Object_Type		@"Type"					// type of variable
#define Object_Index	@"Index"				// index expression - if any
#define Object_Size		@"Size"					// array size - if any
#define Object_Addr		@"Addr"					// symbolic address of object
#define Object_Storage	@"Storage"				// byref (i.e. lval capable) or byval
#define Object_Label	@"Label"				// label used for parameter list entries
#define Object_Default	@"Default"				// default value used for parameter list entries
- (NSDictionary *) scanType;					// scan symbolic data type (for DIM, NEW etc.)
- (NSDictionary *) scanAsType;					// scan array dimensions and AS part with symbolic data type (for DIM, NEW etc.)
- (NSString *) scanIndex;						// get array index expression
- (NSDictionary *) scanArguments:(NSString *) method;	// scan argument list for method call
- (NSDictionary *) scanReference;				// scan symbolic object reference (i.e. object.object or method(args))
- (NSDictionary *) scanObject;					// scan symbolic object (incl. constants)
- (NSDictionary *) lval;						// left side assignment expression
- (NSDictionary *) fetch:(NSDictionary *) c;	// copy/fetch value from container if it is a byref
- (NSDictionary *) cast:(NSDictionary *) c to:(NSString *)type;	// cast to type
- (NSDictionary *) expr;						// evaluate expression
- (NSDictionary *) expr:(unsigned) level;		// evaluate expression of that level

- (void) skipBlanks;		// skip blanks
- (void) need:(NSString *)token error:(NSString *)msg;		// check for token - raise Syntax Error if not found

- (void) contextPush:(id) type;			// start new context
- (void) contextPop;					// end context
- (BOOL) contextIs:(NSString *) type;	// check if current context matches string

@end

@implementation HNSBasicCompiler

static HNSBasicCompiler *shcomp;

+ (HNSBasicCompiler *) sharedCompiler
{
	if(shcomp == nil)
		shcomp=[[[HNSBasicCompiler alloc] init] retain];
	return shcomp;
}

+ (NSDictionary *) keyword:(NSString *) sym
{ // get keyword ressource
	static NSDictionary *kw;
	if(kw == nil)
		{ // load from resources file
		NSString *sympath=[[NSBundle bundleForClass:[self class]] pathForResource:@"HNSBasicKeywords" ofType:@"plist"];
		kw=[[NSDictionary dictionaryWithContentsOfFile:sympath] retain];
#if 0
		NSLog(@"dictionary %@ loaded from %@", sym, sympath);
#endif
		}
	return [kw objectForKey:sym];
}

- (id) init
{ // Basic Compiler
	self=[super init];
	if(self)
		{
		context=[[NSMutableArray arrayWithCapacity:10] retain];				// create context stack
		locals=[[NSMutableDictionary dictionaryWithCapacity:10] retain];	// local variables
		params=[[NSMutableDictionary dictionaryWithCapacity:10] retain];	// parameter variables
		whitespace=[[NSCharacterSet characterSetWithCharactersInString:@" \t"] retain];
#if 0
		NSLog(@"whitespace: %@", whitespace);
#endif
		numerical=[[NSCharacterSet characterSetWithCharactersInString:@".0123456789"] retain];
		symbolical=[[NSCharacterSet
characterSetWithCharactersInString:@"_0123456789$abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"] retain];
		operatorial=[[NSCharacterSet characterSetWithCharactersInString:@"*+-/^<>="] retain];
		monops=[[HNSBasicCompiler keyword:@"MonOps"] retain];	// get entry from keyword ressource
        binops=[[HNSBasicCompiler keyword:@"BinOps"] retain];	// get entry from keyword ressource
		keywords=[[HNSBasicCompiler keyword:@"Keywords"] retain];	// get entry from keyword ressource
#if 0
		NSLog(@"%@", binops);
#endif
		}
#if 0
	NSLog(@"[BasicCompiler init]");
#endif
	return self;
}

- (void) dealloc
{
#if 0
	NSLog(@"[BasicCompiler dealloc]");
#endif
	[source release];
	[context release];
	[locals release];
	[params release];
	[whitespace release];
	[numerical release];
	[symbolical release];
	[operatorial release];
	[binops release];
	[keywords release];
	[super dealloc];
}

- (void) contextPush:(id) type;	// start new context
{
#if 0
	NSLog(@"push: %@", type);
#endif
	[context addObject:type];		// add to stack
}

- (void) contextPop;					// end context
{
#if 0
	NSLog(@"pop: %@", [context lastObject]);
#endif
	[context removeLastObject];
}

- (BOOL) contextIs:(NSString *) type;	// check if current context
{
#if 0
	NSLog(@"contextIs: %@", [context lastObject]);
#endif
	return [[context lastObject] isEqualToString:type];
}

// main methods

+ (unsigned) linenumber;		// for error processing
{
	return shcomp->linenumber;
}

+ (unsigned) errorPos;			// scanner position for last error
{
	if(shcomp->source != nil)
		return [shcomp->source scanLocation];
	return 0;
}

+ (NSString *) source;			// (re)fetch source
{
	return [shcomp->source string];
}

+ (NSString *) code;			// (re)fetch compiled code
{
	return shcomp->code;
}

+ (NSDictionary *) returnType;		// get return type
{
	return shcomp->returnType;
}

+ (NSDictionary *) params;			// get parameter list
{
	return shcomp->params;
}

+ (NSString *) compile:(NSString *) s
{ // compile this source
	[self sharedCompiler];	// create if not yet existing
#if 0
	NSLog(@"compile:\n%@", s);
#endif
	[shcomp->source autorelease];	// release previous scanner
	shcomp->source=[[NSScanner scannerWithString:s] retain];
	[shcomp->source setCaseSensitive:NO];				// ignore case
	[shcomp->source setCharactersToBeSkipped:nil];		// no automatic skipping
	shcomp->linenumber=1;
	[shcomp->context removeAllObjects];
	[shcomp->locals removeAllObjects];
	[shcomp->params removeAllObjects];
	[shcomp->code autorelease];	// release previous scanner
	shcomp->code=[[NSMutableString stringWithCapacity:1000] retain];
	while(![shcomp->source isAtEnd])
		{
		[shcomp skipBlanks];
		// this code should also watch out for comments!!!
		if(![shcomp->source isAtEnd] && ![shcomp->source scanString:@"\n" intoString:nil])	// ignore empty line
			{
			[shcomp scanStatement];	// compile statement
			[shcomp skipBlanks];
			if(![shcomp->source isAtEnd] && ![shcomp->source scanString:@"\n" intoString:nil])	// some error
				[NSException raise:@"Syntax Error" format:@"Comment or end of line expected"];
			}
		shcomp->linenumber++;
		}
	if([shcomp->context count] != 0)
		[NSException raise:@"Context Error" format:@"Unexpected end of source. Unmatched %@", [shcomp->context lastObject]];
#if 0
	NSLog(@"generated code: %@", shcomp->code);
#endif
	return shcomp->code;
}

// scanner/parser

- (void) skipBlanks
{ // skip blanks and comments
	while([source scanCharactersFromSet:whitespace intoString:nil])
		;
	if([source scanString:@"'" intoString:nil] || [source scanString:@"//" intoString:nil])
		{ // advance just before EOL
			[source scanUpToString:@"\n" intoString:nil];
		}
}

- (void) skipToEol
{ // advance to start of next line
	[source scanUpToString:@"\n" intoString:nil];
	[source scanString:@"\n" intoString:nil];
}

- (NSString *) scanName
{ // scan symbolic name - begin with letter, continue with letter/digit
	NSString *n;
	[self skipBlanks];
	if([source scanCharactersFromSet:symbolical intoString:&n])
		{
#if 0
		NSLog(@"symbol: %@", n);
#endif
		return n;
		}
	return nil;
}

- (void) need:(NSString *)token error:(NSString *)msg;		// check for token - raise Syntax Error if not found
{
	if(![[[self scanName] lowercaseString] isEqualToString:token])
		[NSException raise:@"Syntax Error" format:@"%@", msg];
}

- (NSString *) getCocoaType:(NSString *) sym;
{ // convert Basic to Cocoa type
	NSString *lsym=[sym lowercaseString];
	// built-in types and translation
	// should read from resource!
	if([lsym isEqualToString:@"boolean"])
		return @"NSNumber";
	else if([lsym isEqualToString:@"integer"])
		return @"NSNumber";
	else if([lsym isEqualToString:@"double"])
		return @"NSNumber";
	else if([lsym isEqualToString:@"single"])
		return @"NSNumber";
	else if([lsym isEqualToString:@"string"])
		return @"NSString";
	else if([lsym isEqualToString:@"object"])
		return @"NSObject";	// accepts any object
	else if(NSClassFromString(sym) != nil)	// class by name exists
		return sym;
	return nil;	// is no valid type
}

- (NSDictionary *) scanType
{ // scan symbolic data type
	NSString *sym, *ctype;
	NSMutableDictionary *p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"", Object_Storage, @"", Object_Name, @"", Object_Index, nil];
	sym=[self scanName];
	if(sym == nil)
		[NSException raise:@"Syntax Error" format:@"Type name expected"];
	ctype=[self getCocoaType:sym];
	if(ctype == nil)
		[NSException raise:@"Type Error" format:@"Unknown data type '%@'", sym];
	[p setObject:sym forKey:Object_Type];
	[p setObject:ctype forKey:Object_Addr];
	return p;
}

- (BOOL) nameDefined:(NSString *) name
{ // check if symbol is defined (somewhere) - for DIM statement
#if 0
	NSLog(@"nameDefined:%@", name);
	NSLog(@"locals:%@", locals);
	NSLog(@"params:%@", params);
#endif
	if([keywords objectForKey:[name lowercaseString]])
		return YES;	// is a keyword
	if([monops objectForKey:name])	// ------ monops stores name(args)
		return YES;	// is a builtin function
	if([locals objectForKey:name])
		return YES;	// is already defined as a local variable
	if([params objectForKey:name])
		return YES;	// is a defined parameter
	if([self getCocoaType:name])
		return YES;	// is a class name
	return NO;	// no, name is available
}

- (NSDictionary *) scanObject
{ // scan object - and try to return reference if possible instead of value
	// (expr)
	// [value, value, ...] --- initialized NSArray
	// [key:value, key:value, ...]	--- initialized NSDictionary
	// [] --- empty NSArray
	// [:] --- empty NSDictionary
	// new class (arg, ...)
	// builtin-function(args, ...)  --- no space between symbol and (
	// builtin-function args, ...
	// class --- constant
	// class(arg) --- explicit cast
	// constants --- numeric, string, symbolic
	// objects --- locals, paramters, globals
	NSString *sym;
	NSString *ctype;
	NSMutableDictionary *p=nil;
	NSMutableArray *a=[NSMutableArray arrayWithCapacity:5];		// function argument
	id op;	// operator description - either NSString or NSArray
	unsigned pos;		// for backing up
	double d;
	int i;
#if 0
	NSLog(@"scanObject");
#endif
	[self skipBlanks];
	if([source scanString:@"(" intoString:nil])
		{
		NSDictionary *pp=[self expr:0];	// any expression/container - not being fetched here (!)
		[self skipBlanks];
		if(![source scanString:@")" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"Closing ) expected"];
		return pp;
		}
	if([source scanString:@"[" intoString:nil])
		{ // inline array/dictionary
		NSDictionary *pp;
		[self skipBlanks];
		if([source scanString:@":" intoString:nil])
			{ // [:
			[self skipBlanks];
			if(![source scanString:@"]" intoString:nil])
				[NSException raise:@"Syntax Error" format:@"Closing ] after [: expected"];
			return [NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"[:]", Object_Addr, @"NSDictionary", Object_Type, nil];
			}
		if([source scanString:@"]" intoString:nil])
			{ // []
			return [NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"[]", Object_Addr, @"NSArray", Object_Type, nil];
			}
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"[", Object_Addr, @"NSArray", Object_Type, nil];
		while(YES)
			{ // [expr
			pp=[self expr];	// any expression/container
			[p setObject:[[p objectForKey:Object_Addr] stringByAppendingString:[pp objectForKey:Object_Addr]] forKey:Object_Addr];	// append
			[self skipBlanks];
			if([source scanString:@":" intoString:nil])
				{ // [key:value, ...
				[p setObject:@"NSDictionary" forKey:Object_Type];	// change type
				pp=[self expr];	// any expression/container
				[p setObject:[[p objectForKey:Object_Addr] stringByAppendingFormat:@":%@", [pp objectForKey:Object_Addr]] forKey:Object_Addr];
				// make first key:value pair
				while([self skipBlanks], [source scanString:@"," intoString:nil])
					{ // next pair follows
					pp=[self expr];	// any expression/container
					[p setObject:[[p objectForKey:Object_Addr] stringByAppendingFormat:@" %@", [pp objectForKey:Object_Addr]] forKey:Object_Addr];
					[self skipBlanks];
					if(![source scanString:@":" intoString:nil])
						[NSException raise:@"Syntax Error" format:@"Separating : expected between Key and Value"];
					pp=[self expr];	// any expression/container
					[p setObject:[[p objectForKey:Object_Addr] stringByAppendingFormat:@":%@", [pp objectForKey:Object_Addr]] forKey:Object_Addr];
					}
				[self skipBlanks];
				if(![source scanString:@"]" intoString:nil])
					[NSException raise:@"Syntax Error" format:@"Closing ] expected"];
				return p;
				}
			[self skipBlanks];
			if(![source scanString:@"," intoString:nil])
				break;
			[p setObject:[[p objectForKey:Object_Addr] stringByAppendingString:@","] forKey:Object_Addr];	// delimit
			}
		[self skipBlanks];
		if(![source scanString:@"]" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"Closing ] expected"];
		[p setObject:[[p objectForKey:Object_Addr] stringByAppendingString:@"]"] forKey:Object_Addr];	// close
		return p;
		}
	if([source scanString:@"\"" intoString:nil])
		{ // check for "" -> string constant
		NSString *j, *s;	// junk
//		[source setCharactersToBeSkipped:nil];	// nothing
		s=@"";
		while(YES)
			{
			[source scanUpToCharactersFromSet:[NSCharacterSet characterSetWithCharactersInString:@"\"\\\n"] intoString:&j];
			s=[s stringByAppendingString:j];	// append junk
			if([source scanString:@"\"" intoString:nil])
				{ // may be done
				// [self skipBlanks];	// don't allow for "abc" "def" - only "abc""def"
				if([source scanString:@"\"" intoString:nil])
					{ // second " begins
					s=[s stringByAppendingString:@"\""];	// append single quote
					continue;
					}
				break;	// string is closed
				}
			if(![source scanString:@"\\" intoString:nil])
				[NSException raise:@"Syntax Error" format:@"Invalid string (unterminated)"];	// neither " nor \ follows? -> unexpected EOF or EOL
			if([source scanString:@"n" intoString:nil])
				j=@"\n";
			else if([source scanString:@"t" intoString:nil])
				j=@"\t";
			else if([source scanString:@"\"" intoString:&j] || [source scanString:@"\\" intoString:&j])
				; // already stored in j
			else if([source scanString:@"x" intoString:nil])
				;	// 2 digit hex follows
			else if([source scanString:@"u" intoString:nil])
				;	// 4 digit hex (unicode) follows
			else
				continue;	// i.e. ignore escape prefix
			s=[s stringByAppendingString:j];	// append escaped character
			}
#if 0
		NSLog(@"string=%@", p);
#endif
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, [NSString stringWithFormat:@"\"%@\"", s], Object_Addr, @"String", Object_Type, nil];
		return p;
		}
	if([source scanDouble:&d])	// somehow try to ignore single dot not followed by at least one digit!
		{ // numerical constant
#if 0
		NSLog(@"number=%f", d);
#endif
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, nil];
		if(d == (int) d)
			{
			[p setObject:[NSString stringWithFormat:@"%d", (int) d] forKey:Object_Addr];
			[p setObject:@"Integer" forKey:Object_Type];
			}
		else
			{
			[p setObject:[NSString stringWithFormat:@"%lg", d] forKey:Object_Addr];
			[p setObject:@"Double" forKey:Object_Type];
			}
		return p;
		}
	sym=[self scanName];
	if(sym == nil)
		{ // try operator characters
	// !!!!! should allow to interpret a+-b as a+(-b) and not the operator +-
		if(![source scanCharactersFromSet:operatorial intoString:&sym])
			[NSException raise:@"Syntax Error" format:@"Symbol or constant expected: %@...",
				[[source string] substringWithRange:NSMakeRange([source scanLocation], 6)]];
		}
	if([[sym lowercaseString] isEqualToString:@"true"])
		{
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"YES", Object_Addr, @"Boolean", Object_Type, nil];
		return p;
		}
	if([[sym lowercaseString] isEqualToString:@"false"])
		{
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"NO", Object_Addr, @"Boolean", Object_Type, nil];
		return p;
		}
	if([[sym lowercaseString] isEqualToString:@"nil"])
		{
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"nil", Object_Addr, @"Object", Object_Type, nil];
		return p;
		}
	if([[sym lowercaseString] isEqualToString:@"self"])
		{
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byref", Object_Storage, @"", Object_Name, @"", Object_Index, @"self", Object_Addr, @"Object", Object_Type, nil];
		return p;
		}
	if([[sym lowercaseString] isEqualToString:@"super"])
		{
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"super", Object_Addr, @"Object", Object_Type, nil];
		return p;
		}
	if([[sym lowercaseString] isEqualToString:@"app"])
		{
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"app", Object_Addr, @"NSApplication", Object_Type, nil];
		return p;
		}
	if([[sym lowercaseString] isEqualToString:@"new"])
		{ // NEW TYPE (args, ...)
		p=[[[self scanType] mutableCopy] autorelease];	// get class
		[p setObject:[NSString stringWithFormat:@"((%@) alloc %@) autorelease", [p objectForKey:Object_Addr], [[self scanArguments:@"init"] objectForKey:Object_Addr]] forKey:Object_Addr];
		return p;
		}
	p=[params objectForKey:sym];
	if(p)
		return [[p copy] autorelease];	// yes, is a function/sub parameter
	p=[locals objectForKey:sym];
	if(p)
		return [[p copy] autorelease];	// yes, is a locally defined variable
	ctype=[self getCocoaType:sym];	// type name?
	if(ctype)
		{ // class object
		p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, ctype, Object_Addr, sym, Object_Type, nil];
		[self skipBlanks];
		if([source scanString:@"(" intoString:nil])
			{ // cast
			[p setObject:[NSString stringWithFormat:@"(%@) castTo:(%@ class)", [[self expr] objectForKey:Object_Addr], ctype] forKey:Object_Addr];
			if(![source scanString:@")" intoString:nil])
				[NSException raise:@"Syntax Error" format:@"Missing ) in cast operator"];
			}
		return p;
		}
	// check for Cocoa constant e.g. NSASCIIStringEncoding
	pos=[source scanLocation];	// save to allow backing up if symbol follows but is not a valid operator on this level
	op=[monops objectForKey:[sym lowercaseString]];
#if 0
	NSLog(@"monops: %@", op);
#endif
	if(op == nil)
		[NSException raise:@"Syntax Error" format:@"Unknown symbol encountered: %@", sym];
//	[self skipBlanks];
	if([source scanString:@"(" intoString:nil])
		{ // get parenthesized arguments - no space between symbol and (
		while(YES)
			{
			[a addObject:[self expr]];
			[self skipBlanks];
			if(![source scanString:@"," intoString:nil])
				break;
			}
		if(![source scanString:@")" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"Missing ) in function call"];
		}
	else
		{ // get argument list without parenthesis
		while(YES)
			{ // at least one argument
			[a addObject:[self fetch:[self scanReference]]];	// start on higher level, so that "-a+b" is "(-a)+b" and "not a and b" is "not(a) and b"
			[self skipBlanks];
			if(![source scanString:@"," intoString:nil])
				break;
			}
		}
	op=[monops objectForKey:[NSString stringWithFormat:@"%@(%d)", [sym lowercaseString], [a count]]];	// check for proper argument count
#if 0
	NSLog(@"args: %@", a);
	NSLog(@"monops: %@", op);
#endif
	if(op == nil)
		[NSException raise:@"Syntax Error" format:@"Invalid number of arguments for function: %@", sym];
	p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, @"", Object_Name, @"", Object_Index, @"Object", Object_Type, nil];
	[p setObject:[NSString stringWithFormat:@"(%@) %@", [[a objectAtIndex:0] objectForKey:Object_Addr], [op objectAtIndex:0]] forKey:Object_Addr];
#if 0
	NSLog(@"addr: %@", [p objectForKey:Object_Addr]);
#endif
	for(i=1; i<[a count]; i++)
		{ // process method call and argument(s)
		[p setObject:[[p objectForKey:Object_Addr] stringByAppendingFormat:@"(%@)", [[a objectAtIndex:i] objectForKey:Object_Addr]] forKey:Object_Addr];
		if(i<[op count])
			[p setObject:[[p objectForKey:Object_Addr] stringByAppendingFormat:@" %@", [op objectAtIndex:i]] forKey:Object_Addr];	// next method part
		}
#if 0
	NSLog(@"addr: %@", [p objectForKey:Object_Addr]);
#endif
	return p;
}

- (NSDictionary *) scanArguments:(NSString *) method;
{ // scan additional arguments for method and build method call
	unsigned int pos;
	NSString *sym;
	NSMutableDictionary *p=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"byval", Object_Storage, method, Object_Name, @"", Object_Index, nil];
	[self skipBlanks];
	if([source scanString:@"(" intoString:nil] && ([self skipBlanks], ![source scanString:@")" intoString:nil]))
		{ // method call with parameters - and not method()
		while(YES)
			{
			NSDictionary *arg;
			pos=[source scanLocation];	// remember
			sym=[self scanName];
			if(sym && ([self skipBlanks], [source scanString:@":" intoString:nil]))
				{ // named argument follows
				arg=[self expr];
				// could use lval and check for byref/byval passing
				method=[method stringByAppendingFormat:@"%@:(%@)", sym, [arg objectForKey:Object_Addr]];
				}
			else
				{ // unnamed
				[source setScanLocation:pos];	// back up before symbol
				// may try to get data type and say withXXX:
				arg=[self expr];
				// could use lval and check for byref/byval passing
				method=[method stringByAppendingFormat:@":(%@)", [arg objectForKey:Object_Addr]];
				}
			[self skipBlanks];
			if(![source scanString:@"," intoString:nil])
				break;
			}
		if(![source scanString:@")" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"Invalid function arguments"];
		}
	[p setObject:method forKey:Object_Addr];	// method incl. arguments
	return p;
}

- (NSDictionary *) scanReference
{   // handle:
	// reference.component
	// reference.method(arg, ...)
	// array(index, ...)
	NSString *c;
	NSDictionary *p;
	NSMutableDictionary *pp;
	p=[self scanObject];
	while(true)
		{ // check for array index, .component or method call
		[self skipBlanks];
		c=nil;
		if([source scanString:@"(" intoString:nil])
			{ // array
#if 0
			NSLog(@"access indexed: %@", p);
#endif
			if([[p objectForKey:Object_Index] length] != 0)
				p=[self fetch:p];	// fetch first because index is already in use (should not really happen as we don't support arrays of array!)
			pp=[[p mutableCopy] autorelease];
			[pp setObject:[self scanIndex] forKey:Object_Index];	// insert (new) index
			p=pp;
#if 0
			NSLog(@"index added: %@", p);
#endif
			continue;
			}
		if(![source scanString:@"." intoString:nil])
			break;	// neither
		if(c == nil)
			c=[self scanName];	// get component name
		if(c == nil)
			[NSException raise:@"Syntax Error" format:@"Missing component/method name"];
#if 0
		NSLog(@"access component %@ of %@", c, p);
#endif
		p=[self fetch:p];	// fetch first before calling method on array object
		// should allow for keeping p as byref if scanArguments has no arguments -> may go on left side of assignment
		pp=[[p mutableCopy] autorelease];
		[pp setObject:[NSString stringWithFormat:@"(%@) %@", [p objectForKey:Object_Addr], [[self scanArguments:c] objectForKey:Object_Addr]] forKey:Object_Addr];	// call method - may be getter method
		p=pp;
		}
#if 0
	NSLog(@"scanReference: %@", [p objectForKey:Object_Addr]);
#endif
    return p;
}

- (NSDictionary *) expr:(unsigned) level
{ // process expression: (mainly)
  // reference + reference, reference - reference
  // reference * reference, reference / reference
  // reference <= reference, ...
  // reference isa type
	NSDictionary *l, *r;
	NSString *sym, *lcsym;
	NSDictionary *op;
	unsigned pos;
#if 0
	NSLog(@"expr:%d", level);
#endif
	if(level > 6)
		return [self scanReference];	// beyond maximum level
	l=[self expr:level+1];
	while(1)
		{ // check for keywords/operators like isa, and, +, - etc.
		pos=[source scanLocation];	// save to allow backing up if symbol follows but is not a valid operator on this level
		sym=[self scanName];
#if 0
		NSLog(@"level=%d sym=%@", level, sym);
#endif
		if(sym == nil)
			{ // try operator characters
	 // !!!!! should allow to interpret a+-b as a+(-b) and not the operator +-
			if(![source scanCharactersFromSet:operatorial intoString:&sym])
				return l;	// neither
			}
		lcsym=[sym lowercaseString];
		op=[binops objectForKey:lcsym];
		if(op)
			{ // may be an operator
			int ol=[[op objectForKey:@"priority"] intValue];	// operator level
#if 0
			NSLog(@"op %@: %@, ol=%d - level=%d", sym, op, ol, level);
#endif
			if(ol >= level)
				{ // yes, its at least this level
				NSMutableDictionary *ll;
				l=[self fetch:l];	// fetch left value first
				ll=[[l mutableCopy] autorelease];
				if([lcsym isEqualToString:@"isa"])
					{
					r=[self scanType];	// special operand type expected
					[ll setObject:@"Boolean" forKey:Object_Type];
					}
				else
					{
					r=[self fetch:[self expr:ol+1]];	// get right operand which must have higher level than operator
					// set result data type
					// l=[self cast:l to: ];
		            // r=[self cast:r to: ];
					}
#if 0
				NSLog(@"%@.%@(%@)", l, sym, r);
#endif
				[ll setObject:[NSString stringWithFormat:@"(%@) %@(%@)", [l objectForKey:Object_Addr], [op objectForKey:@"selector"], [r objectForKey:Object_Addr]] forKey:Object_Addr];				// glue ll together
				l=ll;
				continue;	// ok, operator and right argument have been processed - same level may follow again
				}
			}
#if 0
		NSLog(@"go back");
#endif
		[source setScanLocation:pos];	// back up and ignore operator
		return l;
		}
}

- (NSDictionary *) expr
{ // generate code for evaluating expression
	return [self fetch:[self expr:0]];	// always return byval
}

- (NSDictionary *) lval
{ // left side expression value - return container
	NSDictionary *v=[self scanReference];	// start at top level (needs () for getting to lower level) - this is because of the ambiguity for the = operator
	if(![[v objectForKey:Object_Storage] isEqualToString:@"byref"])
#if 1
		[NSException raise:@"Value Error" format:@"Assignable value expected (%@)", v];
#else
	[NSException raise:@"Value Error" format:@"Assignable value expected"];
#endif
	return v;
}

- (NSDictionary *) fetch:(NSDictionary *) c
{ // copy/fetch value from container
	NSMutableDictionary *nc;
	if([[c objectForKey:Object_Size] length] != 0 && [[c objectForKey:Object_Index] length] == 0)
		{ // array object without index
		return c;
		}
	if([[c objectForKey:Object_Storage] isEqualToString:@"byref"])
		{ // must fetch by pointer
		nc=[[c mutableCopy] autorelease];
		if([[c objectForKey:Object_Index] length] != 0)
			{ // fetch per index
	 // check for single-dimension and use objectAtIndex: method
			[nc setObject:[[c objectForKey:Object_Addr] stringByAppendingFormat:@" objectAtIndexList:[%@]", [c objectForKey:Object_Index]] forKey:Object_Addr];
			[nc setObject:@"" forKey:Object_Index];	// remove indexing
			}
		else
			[nc setObject:[[c objectForKey:Object_Addr] stringByAppendingFormat:@" object"] forKey:Object_Addr];
		[nc setObject:@"byval" forKey:Object_Storage];	// make byval
		c=nc;
		}
	// check byval and index???
	return c;
}

- (NSString *) scanParameterList:(BOOL) cls
{ // scan function name and optional parameter list
	NSString *methodName;
#if 0
	NSLog(@"scanParameterList");
#endif
	methodName=[self scanName];
	if(!methodName)
		[NSException raise:@"Syntax Error" format:@"Missing method name"];
	if(cls)
		methodName=[NSString stringWithFormat:@"+%@", methodName];	// class method
	else
		methodName=[NSString stringWithFormat:@"-%@", methodName];	// instance method
	[self skipBlanks];
	[params removeAllObjects];	// make empty list
	if([source scanString:@"(" intoString:nil] && ([self skipBlanks], ![source scanString:@")" intoString:nil]))
		{ // argument list follows
		while(YES)
			{ // [label:] [byref | byval] var as type [=default] [, ...]
			NSString *lbl=nil;
			NSString *var;
			NSMutableDictionary *c;
			NSString *ref;
			unsigned pos;
			pos=[source scanLocation];	// save for backing up if no label follows
			lbl=[self scanName];
			[self skipBlanks];
			if(!lbl || ![source scanString:@":" intoString:nil])
				{ // no label:
				lbl=nil;
				[source setScanLocation:pos];	// restore initial scan position
				}
			var=[self scanName];
			if(var && [[var lowercaseString] isEqualTo:@"byref"])
				ref=@"byref", var=[self scanName];
			else if(var && [[var lowercaseString] isEqualTo:@"byval"])
				ref=@"byval", var=[self scanName];
			else
				ref=@"byval";
			if(!var)
				[NSException raise:@"Syntax Error" format:@"Missing parameter name"];
			if([self nameDefined:var])
				[NSException raise:@"Symbol Error" format:@"Parameter name '%@' is already in use", var];
			[self need:@"as" error:@"Missing AS part in parameter list"];
			c=[[[self scanType] mutableCopy] autorelease];
			[c setObject:ref forKey:Object_Storage];	// set storage
			if(lbl)
				{
				[c setObject:lbl forKey:Object_Label];	// save label
				methodName=[methodName stringByAppendingString:lbl];	// append label
				}
			else
				; // may define lbl as withClass
			methodName=[methodName stringByAppendingString:@":"];		// prefix for parameter
			if([source scanString:@"=" intoString:nil])
				[c setObject:[self expr] forKey:Object_Default];
			[c setObject:[NSString stringWithFormat:@"argv[%d]", [params count]+2] forKey:Object_Addr];	// access parameter
			[params setObject:c forKey:var];	// add to parameter list
			[self skipBlanks];
			if(![source scanString:@"," intoString:nil])
				break;
			}
		[self skipBlanks];
		if(![source scanString:@")" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"Missing ) to end parameter list"];
		}
#if 0
	NSLog(@"method: %@", methodName);
#endif
	return methodName;
}

- (NSString *) scanIndex;
{ // opening ( already assumed
	NSString *i=@"";
	while(YES)
		{ // get index list
		NSDictionary *a=[self expr];
		// check for constant and constants must be positive
		i=[i stringByAppendingFormat:@"%@", [a objectForKey:Object_Addr]];
		[self skipBlanks];
		if([source scanString:@"," intoString:nil])
			{
			i=[i stringByAppendingFormat:@","];
			continue;
			}
		if([source scanString:@")" intoString:nil])
			break;
		[NSException raise:@"Syntax Error" format:@"Missing ) in DIM or REDIM index list"];
		}
	return i;
}

- (void) assign:(NSDictionary *) l value:(NSDictionary *) r;
{ // create assignment
	if(![[l objectForKey:Object_Storage] isEqualToString:@"byref"])
#if 1
		[NSException raise:@"Value Error" format:@"Assignable value expected (%@)", l];
#else
	[NSException raise:@"Value Error" format:@"Assignable value expected"];
#endif
	if([[l objectForKey:Object_Index] length] != 0)
		{ // array assignment
		// check for single-dimensional array
		[code appendFormat:@"%@ replaceObjectAtIndexList:[%@] withObject:(%@);\n", [l objectForKey:Object_Addr], [l objectForKey:Object_Index], [r objectForKey:Object_Addr]];
		}
	else
		[code appendFormat:@"%@ replaceObject:(%@);\n", [l objectForKey:Object_Addr], [r objectForKey:Object_Addr]];
}

- (void) scanStatement
{ // execute single token of Basic source; may be line or next part of expression
	NSDictionary *l, *r;
	NSString *sym;
	BOOL lcls;
	if(0 /* breakpoint on this line */)
		[code appendFormat:@"{script bkpt:(%d)}; ", linenumber];
	else if(0 /* making code for debugging */)
		[code appendFormat:@"{script stop:(%d)}; ", linenumber];	// potential stopping point for single stepper
	linestart=[source scanLocation];	// save start of this line for backing up assignments
	sym=[[self scanName] lowercaseString];	// may be nil!
#if 0
	NSLog(@"scanStatement: %@", sym);
#endif
	if([[context lastObject] isEqualToString:@"Select"] && [[context objectAtIndex:[context count]-2] length] == 0)
		{ // only some statements allowed!
		if(!([sym isEqualToString:@"case"] || [sym isEqualToString:@"else"] || [sym isEqualToString:@"end"]))
			[NSException raise:@"Syntax Error" format:@"Invalid statement following SELECT"];
		}
	if(lcls=[sym isEqualToString:@"class"])
		{ // class prefix
		sym=[[self scanName] lowercaseString];	// get next word
		if(![sym isEqualToString:@"sub"] && ![sym isEqualToString:@"function"])
			[NSException raise:@"Syntax Error" format:@"CLASS not in SUB or FUNCTION context"];
		}
	if([sym isEqualToString:@"sub"])
		{ // SUB name(args, ...) - start subroutine
		if(linenumber != 1)
			[NSException raise:@"Syntax Error" format:@"SUB statement not on first line"];
		[self contextPush:@"Sub"];		// push
		sym=[self scanParameterList:lcls];		// parse name and local parameters
		[returnType autorelease];
		returnType=nil;					// no return type
		[code appendFormat:@"HNSCocoaScript defineMethod:(\"%@\") ofType:(nil) andArgs:[] asScript:{{\n", sym];
		return;
		}
	if([sym isEqualToString:@"function"])
		{ // FUNCTION name(args, ...) AS type - start function
		if(linenumber != 1)
			[NSException raise:@"Syntax Error" format:@"FUNCTION statement not on first line"];
		[self contextPush:@"Function"];	// push
		sym=[self scanParameterList:lcls];		// parse name and local parameters
		[self need:@"as" error:@"Missing AS part in FUNCTION statement"];
		[returnType autorelease];
		returnType=[[self scanType] retain];
		[code appendFormat:@"HNSCocoaScript defineMethod:(\"%@\") ofType:(%@) andArgs:[] asScript:{{\n", sym, [self getCocoaType:[returnType objectForKey:Object_Type]]];
		return;
		}
	if(linenumber == 1)
		[NSException raise:@"Syntax Error" format:@"missing FUNCTION or SUB statement on first line"];
	if([sym isEqualToString:@"end"])
		{ // END statement
		sym=[[self scanName] lowercaseString];
		if([[context lastObject] isEqualToString:@"Select"] && [[context objectAtIndex:[context count]-2] length] == 0)
			{ // only some statements allowed!
			if(!([sym isEqualToString:@"select"]))
				[NSException raise:@"Syntax Error" format:@"Invalid END statement following SELECT"];
			}
		if([sym isEqualToString:@"if"])
			{ // END IF
			int n;
			if(![self contextIs:@"If"] && ![self contextIs:@"IfElse"])
				[NSException raise:@"Context Error" format:@"END IF not in IF context"];
			[self contextPop];
			n=[[context lastObject] intValue];	// get counter
			while(n-- >0)
				[code appendString:@"}"];	// close additional {
			[self contextPop];	// elseif counter
			[code appendString:@"};\n"];
			return;
			}
		if([sym isEqualToString:@"sub"])
			{ // END SUB
			if(![self contextIs:@"Sub"])
				// could print relevant context, i.e. IF context not closed
				// for that, contextIs should be replaced by contextNeeded:@"Sub" message:@"END SUB not in correct context"
				[NSException raise:@"Context Error" format:@"END SUB not in correct context"];
			[self contextPop];
			[code appendFormat:@"} CSfunction }\n"];	// return nil
			return;
			}
		if([sym isEqualToString:@"function"])
			{ // END FUNCTION
			if(![self contextIs:@"Function"])
				[NSException raise:@"Context Error" format:@"END FUNCTION not in correct context"];
			[self contextPop];
			[code appendFormat:@"} CSfunction }\n"];	// return value
			return;
			}
		if([sym isEqualToString:@"select"])
			{ // END SELECT
			if([self contextIs:@"SelectElse"])
				{ // close SELECT ELSE part
				[code appendFormat:@"};\n"];	// CSSelect:[...] DEFAULT:{...};
				[self contextPop];
				[self contextPop];
				return;
				}
			if(![self contextIs:@"Select"])
				[NSException raise:@"Syntax Error" format:@"END SELECT not in SELECT context"];
			if([[context objectAtIndex:[context count]-2] length] != 0)
				{ // at least one case
				[code appendFormat:@"}];\n"];	// close last statement and close list
				}
			else
				{ // there was NO case at all -> empty dictionary
				[code appendFormat:@":];\n"];	// CSSelect:[:];
				}
			[self contextPop];
			[self contextPop];
			return;
			}
		if([sym isEqualToString:@"try"])
			{ // END TRY
			if([self contextIs:@"Try"])
				[NSException raise:@"Syntax Error" format:@"Missing CATCH in TRY context"];
			if(![self contextIs:@"TryCatch"])
				[NSException raise:@"Syntax Error" format:@"END TRY not in TRY context"];
			[self contextPop];
			[code appendFormat:@"};\n"];	// close block
			return;
			}
		if(sym)
			[NSException raise:@"Syntax Error" format:@"Unknown END statement: END %@", sym];
		else
			[NSException raise:@"Syntax Error" format:@"Incomplete END statement"];
		return;
		}
	if([context count] == 0)
		[NSException raise:@"Syntax Error" format:@"Code after END SUB or END FUNCTION"]; // check for context stack underrun, i.e. line coming after END SUB etc.
	if([sym isEqualToString:@"if"])
		{ // IF expr THEN - statement
		[self contextPush:[NSNumber numberWithInt:0]];	// elseif counter
		[self contextPush:@"If"];		// push
		l=[self expr];	// get expression
		[self need:@"then" error:@"Missing THEN in IF statement"];
		// may handle single line IF statement
		[code appendFormat:@"(%@) CSIF:{\n", [l objectForKey:Object_Addr]];
		return;
		}
	if([sym isEqualToString:@"else"])
		{ // ELSE statement
		if([self contextIs:@"IfElse"] || [self contextIs:@"SelectElse"])
			[NSException raise:@"Context Error" format:@"Multiple ELSE in IF or SELECT context"];
		if([self contextIs:@"If"])
			{ // ELSE in IF context
			[self contextPop], [self contextPush:@"IfElse"];	// change context to so that double ELSE is not permitted
			[code appendString:@"}ELSE:{\n"];
			}
		else if([self contextIs:@"Select"])
			{
			[self contextPop], [self contextPush:@"SelectElse"];	// replace
			if([[context objectAtIndex:[context count]-2] length] != 0)
				{ // after at least one case
				[code appendFormat:@"}] DEFAULT:{"];	// close last dictionary entry
				}
			else
				{ // there was NO case at all -> empty dictionary
				[code appendFormat:@":] DEFAULT:{"];	// CSSelect:[:] default:{...};
				}
			}
		else
			[NSException raise:@"Context Error" format:@"ELSE not in IF or SELECT context"];
		return;
		}
	if([sym isEqualToString:@"elseif"])
		{ // ELSEIF expr THEN - statement
		int c;
		if([self contextIs:@"IfElse"])
			[NSException raise:@"Context Error" format:@"ELSEIF after ELSE in IF context"];
		if(![self contextIs:@"If"])
			[NSException raise:@"Context Error" format:@"ELSEIF not in IF context"];
		l=[self expr];	// get condition
		[self need:@"then" error:@"Missing THEN in ELSEIF statement"];
		// may handle single line IF statement
		[code appendFormat:@"}ELSE:{ (%@) CSIF:{\n", [l objectForKey:Object_Addr]];
		c=[[context objectAtIndex:[context count]-2] intValue]+1;	// increment counter
		[context replaceObjectAtIndex:[context count]-2 withObject:[NSNumber numberWithInt:c]];
#if 0
		NSLog(@"elseif counter=%@", [context objectAtIndex:[context count]-2]);
#endif
		return;
		}
	if([sym isEqualToString:@"for"])
		{ // FOR var=start [DOWN]TO end [STEP x] - start counted loop
		NSMutableDictionary *ss;
		BOOL downto;
		unsigned int pos;	// to handle downto/to
		pos=[source scanLocation];
		sym=[[self scanName] lowercaseString];
		if(sym && [sym isEqualTo:@"each"])
			{ // handle FOR EACH var IN array
			l=[self lval];	// loop variable
			[self need:@"in" error:@"Missing IN part in FOR EACH statement"];
			r=[self expr];	// get array to loop through
			[NSException raise:@"Syntax Error" format:@"FOR EACH not yet implemented"];
			}
		[source setScanLocation:pos];	// restore
		l=[self lval];	// loop variable
		if(![[l objectForKey:Object_Storage] isEqualToString:@"byref"])
			[NSException raise:@"Value Error" format:@"Can't assign to object specified as loop variable"];
		// check variable type
		// check for simple variable
		// check for being already used in outer loop
		if(![source scanString:@"=" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"Missing = part in FOR statement"];
		[self assign:l value:[self expr]];	// initialize
		pos=[source scanLocation];
		sym=[[self scanName] lowercaseString];
		downto=((sym != nil) && [sym isEqualTo:@"downto"]);
		if(!downto)
			{ // needs 'to'
			[source setScanLocation:pos];	// restore
			[self need:@"to" error:@"Missing TO or DOWNTO part in FOR statement"];
			}
		r=[self expr];	// get loop limit
		if(downto)
			[code appendFormat:@"{ (%@)>=(%@) CSIF:{\n", [[self fetch:l] objectForKey:Object_Addr], [r objectForKey:Object_Addr]];
		else
			[code appendFormat:@"{ (%@)<=(%@) CSIF:{\n", [[self fetch:l] objectForKey:Object_Addr], [r objectForKey:Object_Addr]];
		sym=[[self scanName] lowercaseString];
		if(sym && ![sym isEqualToString:@"step"])
			[NSException raise:@"Syntax Error" format:@"Missing STEP part in FOR statement"];
		if(sym)
			ss=[[[self expr] mutableCopy] autorelease];	// get step value
		else
			ss=[NSMutableDictionary dictionaryWithObjectsAndKeys:@"1", Object_Addr, @"Integer", Object_Type, @"byval", Object_Storage, nil];
		if(downto)
			[ss setObject:[NSString stringWithFormat:@"-(%@)", [ss objectForKey:Object_Addr]] forKey:Object_Addr];	// i.e. subtract
		else
			[ss setObject:[NSString stringWithFormat:@"+(%@)", [ss objectForKey:Object_Addr]] forKey:Object_Addr];	// i.e. add
		[self contextPush:ss];			// step value
		[self contextPush:l];			// loop variable
		[self contextPush:@"For"];		// push
		return;
		}
	if([sym isEqualToString:@"next"])
		{ // NEXT - close FOR loop
		NSMutableDictionary *ss;
		if(![self contextIs:@"For"])
			[NSException raise:@"Context Error" format:@"NEXT not in FOR context"];
		[self contextPop];
		l=[context lastObject];	// loop variable
		[self contextPop];
		[code appendFormat:@"}ELSE:{NSException raise:(\"HNSBreakLoop\") format:(\"\")}; "];
		ss=[[[self fetch:l] mutableCopy] autorelease];
		[ss setObject:[NSString stringWithFormat:@"%@%@", [ss objectForKey:Object_Addr], [[context lastObject] objectForKey:Object_Addr]] forKey:Object_Addr];	// add/sub step value
		[self assign:l value:ss]; 
		[code appendFormat:@"} CSloop;\n"];
		[self contextPop];
		return;
		}
	if([sym isEqualToString:@"while"])
		{ // WHILE expr  - statement
		[self contextPush:@"While"];		// push
		l=[self lval];	// loop condition
		[code appendFormat:@"{ (%@) CSIF:{\n", [l objectForKey:Object_Addr]];
		return;
		}
	if([sym isEqualToString:@"wend"])
		{ // wend - close WHILE loop
		if(![self contextIs:@"While"])
			[NSException raise:@"Context Error" format:@"WEND not in WHILE context"];
		[self contextPop];
		[code appendString:@"\n}ELSE:{NSException raise:(\"HNSBreakLoop\") format:(\"\")} } CSloop;\n"];
		return;
		}
	if([sym isEqualToString:@"do"])
		{ // DO [UNTIL expr] - statement
		sym=[[self scanName] lowercaseString];
		if(sym && [sym isEqualToString:@"until"])
			{ // do until... loop
			[self contextPush:@"DoUntil"];		// push
			[code appendFormat:@"{ (%@) CSIF:{NSException raise:(\"HNSBreakLoop\") format:(\"\")};", [[self expr] objectForKey:Object_Addr]];
			}
		else
			{ // do ... loop until
			[self contextPush:@"Do"];		// push
			[code appendFormat:@"{"];
			}
		return;
		}
	if([sym isEqualToString:@"loop"])
		{ // LOOP [UNTIL expr] - statement
		if([[context lastObject] isEqualToString:@"DoUntil"])
			{ // close loop
			[code appendString:@"} CSloop;\n"];
			[self contextPop];
			return;
			}
		if(![self contextIs:@"Do"])
			[NSException raise:@"Context Error" format:@"LOOP not in DO context"];
		[self contextPop];
		[self need:@"until" error:@"Missing UNTIL in LOOP statement"];
		[code appendFormat:@"(%@) CSIF:{NSException raise:(\"HNSBreakLoop\") format:(\"\")} } CSloop;", [[self expr] objectForKey:Object_Addr]];
		return;
		}
	if([sym isEqualToString:@"select"])
		{ // SELECT CASE expr - start selection
		[self contextPush:@""];				// no case statement yet
		[self contextPush:@"Select"];		// push
		sym=[[self scanName] lowercaseString];
		if(![sym isEqualToString:@"case"])
			[NSException raise:@"Context Error" format:@"Missing CASE in SELECT statement"];
		l=[self expr];	// select condition
		[code appendFormat:@"(%@) CSSELECT:[", [l objectForKey:Object_Addr]];	// start selection
		return;
		}
	if([sym isEqualToString:@"case"])
		{ // CASE expr - selection case
		if([[context lastObject] isEqualToString:@"SelectElse"])
			[NSException raise:@"Context Error" format:@"CASE after ELSE in SELECT context"];
		if(![self contextIs:@"Select"])
			[NSException raise:@"Context Error" format:@"CASE not in SELECT context"];
		l=[self expr];	// select condition
		if([[context objectAtIndex:[context count]-2] length] != 0)
			{ // second or later case
			[code appendFormat:@"},(%@):{", [l objectForKey:Object_Addr]];	// close prevoius and start next case element
			}
		else
			{ // first case
			[context replaceObjectAtIndex:[context count]-2 withObject:@"case"];	// there was a case statement
			[code appendFormat:@"(%@):{", [l objectForKey:Object_Addr]];	// start first case element
			}
		return;
		}
	if([sym isEqualToString:@"try"])
		{ // TRY - start exception block
		[self contextPush:@"Try"];		// push
		[code appendString:@"{\n"];		// start block
		return;
		}
	if([sym isEqualToString:@"exception"])
		{ // EXCEPTION [var [AS expression]] - exception handler
		if([self contextIs:@"TryCatch"])
			[NSException raise:@"Context Error" format:@"Multiple CATCH in TRY context"];
		if(![self contextIs:@"Try"])
			[NSException raise:@"Context Error" format:@"CATCH not in TRY context"];
		[self contextPop], [self contextPush:@"TryCatch"];	// replace
		// allow for missing expression
		l=[self lval];	// variable
		[code appendFormat:@" } CSexception:{ %@:=argv[2];\n", [l objectForKey:Object_Addr]];
		return;
		}
// other builtin statements
	if([sym isEqualToString:@"dim"])
		{ // DIM var, ... AS type, ... statement
		if(![self contextIs:@"Sub"] && ![self contextIs:@"Function"])
			[NSException raise:@"Context Error" format:@"DIM in invalid context"];
		do
			{ // collect name, name, ... AS type , ...
			NSMutableArray *v=[NSMutableArray arrayWithCapacity:5];	// list of variables/indices to be dim'ed
			NSDictionary *type;			// current type
			NSMutableDictionary *var;	// newly created variable
			int i;
			do
				{ // collect names until AS
				NSMutableDictionary *e;
				sym=[self scanName];
				if(sym == nil)
					[NSException raise:@"Syntax Error" format:@"Invalid variable name in DIM statement"];
				[self skipBlanks];
				e=[NSMutableDictionary dictionaryWithObjectsAndKeys:sym, Object_Name, @"", Object_Size, nil];
				if([source scanString:@"(" intoString:nil])	// array
					[e setObject:[self scanIndex] forKey:Object_Size];
				[v addObject:e];
				} while([source scanString:@"," intoString:nil]);
			[self need:@"as" error:@"Missing AS part in DIM statement"];
			type=[self scanType];	// get AS type
			for(i=0; i<[v count]; i++)
				{ // process all entries in array v and initialize them
				var=[[type mutableCopy] autorelease];	// create new entry with that type
				if([self nameDefined:[[v objectAtIndex:i] objectForKey:Object_Name]])	// check here because 'dim i,j,i' is also invalid
					[NSException raise:@"Semantic Error" format:@"Name %@ is already defined", [[v objectAtIndex:i] objectForKey:Object_Name]];
				[var setObject:[NSString stringWithFormat:@"_%@", [[v objectAtIndex:i] objectForKey:Object_Name]] forKey:Object_Addr];		// insert name as address
				[var setObject:[[v objectAtIndex:i] objectForKey:Object_Size] forKey:Object_Size];	// insert index
				[var setObject:@"byref" forKey:Object_Storage];
#if 0
				NSLog(@"%@:=0;", [[v objectAtIndex:i] objectForKey:Object_Name]);
#endif
				[code appendFormat:@"%@:=(HNSTypedTensor alloc ", [var objectForKey:Object_Addr]];
				if([[var objectForKey:Object_Size] length] != 0)
					[code appendFormat:@"initWithDimension:[%@] forClass:", [var objectForKey:Object_Size]];	// array
				else
					[code appendFormat:@"initScalarForClass:"];
				[code appendFormat:@"(%@ class)) autorelease;\n", [self getCocoaType:[var objectForKey:Object_Type]]];
				[locals setObject:var forKey:[[v objectAtIndex:i] objectForKey:Object_Name]];	// create personal dictionary entry for symbol recording its type
#if 0
				NSLog(@"locals: %@", locals);
#endif
				}
			} while([source scanString:@"," intoString:nil]);
		return;
		}
	if([sym isEqualToString:@"redim"])
		{ // REDIM array(size, ...) statement - resize array
		l=[self lval];
		// check for real lval
		[self skipBlanks];
		if(![source scanString:@"(" intoString:nil])
			[NSException raise:@"Syntax Error" format:@"Missing ( in REDIM statement"];
		[code appendFormat:@"(%@) setRho:[%@];\n", [l objectForKey:Object_Addr], [self scanIndex]];
		return;
		}
	if([sym isEqualToString:@"return"])
		{ // RETURN [expr] - return fron sub or function
		if(returnType)
			{ // expression must follow
			l=[self expr];
#if HASCAST
			[code appendFormat:@"retval:=(%@) castTo:(%@);\n", [l objectForKey:Object_Addr], [returnType objectForKey:Object_Type]];
#else
			[code appendFormat:@"retval:=(%@);\n", [l objectForKey:Object_Addr], [returnType objectForKey:Object_Type]];
#endif
			}
		[code appendString:@"NSException raise:(\"HNSReturn\") format:(\"\");\n"];	// function/subroutine exit
		// somehow detect and warn about dead code after RETURN
		return;
		}
	if([sym isEqualToString:@"exit"])
		{ // EXIT - exit loop
		// may be valid in lower level - i.e. go down all If and Select levels
//		if(![self contextIs:@"For"] && ![self contextIs:@"While"] && ![self contextIs:@"Loop"])
//			[NSException raise:@"Syntax Error" format:@"EXIT not in a valid loop context"];
		[code appendString:@"NSException raise:(\"HNSBreakLoop\") format:(\"\");\n"];	// loop exit
		return;
		}
	if([sym isEqualToString:@"call"])
		{ // CALL expr - ignore result
		l=[self expr];
		[code appendFormat:@"%@;\n", [l objectForKey:Object_Addr]];	// ignore result
		return;
		}
	if([sym isEqualToString:@"raise"])
		{ // raise "exception name", "paramter string" - throw exception
		l=[self expr];
		if([source scanString:@"," intoString:nil])
			r=[self expr];
		else
			r=[NSDictionary dictionaryWithObjectsAndKeys:@"\"\"", Object_Addr, nil];	// empty string
		[code appendFormat:@"NSException raise:(%@) format:(%@) arg:(%@);\n", [l objectForKey:Object_Addr], @"\"%@\"", [r objectForKey:Object_Addr]];
		return;
		}
	if([sym isEqualToString:@"const"])
		{ // const symbol=expr
		[NSException raise:@"Syntax Error" format:@"Not yet implemented"];
		}
	if([sym isEqualToString:@"goto"])
		{ // easter-egg 1
		[NSException raise:@"Dijkstra Error" format:@"Go To Statement Considered Harmful\n\nEdsger W. Dijkstra, Communications of the ACM, Vol. 11, No. 3, March 1968, pp. 147-148\nhttp://www.acm.org/classics/oct95/"];
		}
	if([sym isEqualToString:@"peace"])
		{ // easter-egg 2
		[self need:@"forever" error:@"Missing FOREVER part"];
		[NSException raise:@"Historical Error" format:@"This function has been abandoned"];
		}
	// try assignment or method call
	[source setScanLocation:linestart];	// restore initial scan position for trying assignment or procedure call
	l=[self scanReference];	// start at top level (needs () for getting to lower level) - this is because of the ambiguity for the = operator
	if([source scanString:@"=" intoString:nil])
		{ // assignment: l=r
		[self assign:l value:[self expr]];
		return;
		}
#if 0
	NSLog(@"%@", l);
#endif
	// check if value can be ignored - because it was a subroutine call
	// then ok, otherwise error
	[code appendFormat:@"%@;\n", [l objectForKey:Object_Addr]];	// expression as is (should be method call)
}

@end